/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.readers;

import javax.xml.stream.XMLStreamException;

import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;

import eu.europa.ec.taxud.cesop.domain.TransactionDateEnum;
import eu.europa.ec.taxud.cesop.domain.XmlReportedTransaction;
import eu.europa.ec.taxud.cesop.domain.XmlTypeAndValue;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.KEY;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;

import static eu.europa.ec.taxud.cesop.utils.LangUtils.isNotBlank;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.XML_DATE_TIME_FORMATTER;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.convertGreece;

/**
 * Utils class reading a ReportedTransaction XML file.
 */
public class ReportedTransactionXmlReader {

    private static final String PAYMENT_METHOD_NONE = "None";

    private final CesopXmlReader cesopXmlReader;
    private final List<String> otherPaymentMethods;
    private final List<String> otherPspRoles;

    /**
     * Constructor.
     *
     * @param cesopXmlReader      the XML reader.
     * @param otherPaymentMethods the other payment methods
     * @param otherPspRoles       the other psp roles
     */
    public ReportedTransactionXmlReader(final CesopXmlReader cesopXmlReader, final List<String> otherPaymentMethods,
            final List<String> otherPspRoles) {
        this.cesopXmlReader = cesopXmlReader;
        this.otherPaymentMethods = otherPaymentMethods;
        this.otherPspRoles = otherPspRoles;
    }

    private static boolean parseBoolean(String value) {
        return Boolean.parseBoolean(value) || "1".equals(value);
    }

    /**
     * Parses the next reported transaction.
     *
     * @return a ReportedTransaction
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public XmlReportedTransaction parse() throws XMLStreamException {
        if (!this.cesopXmlReader.positionCursorOnStartElement(XML.REPORTED_TRANSACTION_QNAME)) {
            throw new XMLStreamException("Tag " + XML.REPORTED_TRANSACTION_QNAME + " not found.");
        }
        final XmlReportedTransaction xmlReportedTransaction = new XmlReportedTransaction();
        xmlReportedTransaction.setRefund(this.parseIsRefund());
        xmlReportedTransaction.setTransactionIdentifier(this.parseTransactionIdentifier());
        xmlReportedTransaction.setCorrTransactionIdentifier(this.parseCorrTransactionIdentifier());
        if (xmlReportedTransaction.isRefund()) {
            String corrTransactionId = xmlReportedTransaction.getCorrTransactionIdentifier();
            String transactionId = xmlReportedTransaction.getTransactionIdentifier();
            if (corrTransactionId == null || corrTransactionId.trim().isEmpty()) {
                xmlReportedTransaction.setCorrTransactionIdentifier(xmlReportedTransaction.getTransactionIdentifier());
                xmlReportedTransaction.setTransactionIdentifier(generateRefundIdentifier(transactionId));
            } else if (xmlReportedTransaction.getTransactionIdentifier().equals(xmlReportedTransaction.getCorrTransactionIdentifier())) {
                xmlReportedTransaction.setTransactionIdentifier(generateRefundIdentifier(transactionId));
            }
        }
        this.parseDates(xmlReportedTransaction);
        xmlReportedTransaction.setAmount(this.parseAmount());
        this.parsePaymentMethod(xmlReportedTransaction);
        xmlReportedTransaction.setInitiatedPhysical(this.parseInitiatedPhysical());
        xmlReportedTransaction.setPayerMs(this.parsePayerMs());
        this.parsePspRole(xmlReportedTransaction);
        return xmlReportedTransaction;
    }

    /**
     * Generates an identifier for a refund transaction. This is needed when the original transaction identifier
     * is used as the refund identifier or when the refund identifier is empty.
     * The generated identifier is of the form REF_{random number}_{original transaction identifier}
     *
     * @param transactionIdentifier the original transaction identifier
     * @return the generated refund identifier
     */
    private String generateRefundIdentifier(String transactionIdentifier) {
        Random random = new Random();
        String newIdentifier = "REF_" + random.nextInt(100000) + "_" + transactionIdentifier;
        return newIdentifier.substring(0, Math.min(newIdentifier.length(), 100));

    }

    private boolean parseIsRefund() throws XMLStreamException {
        final Map<String, String> attributes = this.cesopXmlReader.getXmlStreamReaderWrapper().getAttributes();
        return parseBoolean(attributes.get(XML.ATTRIBUTE_NAME_IS_REFUND));
    }

    private String parseTransactionIdentifier() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.TRANSACTION_IDENTIFIER_QNAME);
        return valuesMap.get(KEY.TRANSACTION_IDENTIFIER_KEY);
    }

    private String parseCorrTransactionIdentifier() throws XMLStreamException {
        final Optional<Map<String, String>> valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.CORR_TRANSACTION_IDENTIFIER_QNAME);
        return valuesMap.map(m -> m.get(KEY.CORR_TRANSACTION_IDENTIFIER_KEY)).orElse(null);
    }

    private void parseDates(final XmlReportedTransaction xmlReportedTransaction) throws XMLStreamException {
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.DATE_TIME_QNAME)).isPresent()) {
            final String value = valuesMap.get().get(KEY.DATE_TIME_KEY);
            final String type = valuesMap.get().get(KEY.DATE_TIME_TYPE_KEY);
            final String other = valuesMap.get().get(KEY.DATE_TIME_OTHER_KEY);
            final TransactionDateEnum transactionDateEnum = TransactionDateEnum.findByLabel(type);

            // Validate timestamp
            try {
                XML_DATE_TIME_FORMATTER.parse(value);
            } catch (DateTimeParseException e) {
                throw new CesopParsingException("Error while reading the XML file: " + e.getMessage(), e);
            }

            xmlReportedTransaction.addDate(transactionDateEnum, value, other);
        }
    }

    private XmlTypeAndValue parseAmount() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.AMOUNT_QNAME);
        final String amount = valuesMap.get(KEY.AMOUNT_KEY);
        // DB column is NUMBER(17,2)
        if (amount.split("\\.")[0].length() > 15) {
            throw new CesopParsingException(
                    String.format("Error while reading the XML file: The amount %s is too large. Amounts must be less than 10^15.", amount)
            );
        }
        return new XmlTypeAndValue(valuesMap.get(KEY.AMOUNT_CURRENCY_KEY), amount);
    }

    private void parsePaymentMethod(final XmlReportedTransaction xmlReportedTransaction) throws XMLStreamException {
        final Optional<Map<String, String>> valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.PAYMENT_METHOD_QNAME);
        if (valuesMap.isPresent()) {
            final String paymentMethodType = valuesMap.get().get(KEY.PAYMENT_METHOD_TYPE_KEY);
            final String paymentMethodOther = valuesMap.get().get(KEY.PAYMENT_METHOD_OTHER_KEY);
            xmlReportedTransaction.setPaymentMethodType(paymentMethodType);
            if (isNotBlank(paymentMethodOther)) {
                final int index = this.otherPaymentMethods.indexOf(paymentMethodOther);
                if (index == -1) {
                    this.otherPaymentMethods.add(paymentMethodOther);
                    xmlReportedTransaction.setPaymentMethodOtherId(this.otherPaymentMethods.size());
                } else {
                    xmlReportedTransaction.setPaymentMethodOtherId(index + 1);
                }
            }
        } else {
            xmlReportedTransaction.setPaymentMethodType(PAYMENT_METHOD_NONE);
        }
    }

    private boolean parseInitiatedPhysical() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.INITIATED_PHYSICAL_QNAME);
        return parseBoolean(valuesMap.get(KEY.INITIATED_PHYSICAL_KEY));
    }

    private XmlTypeAndValue parsePayerMs() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.PAYER_MS_QNAME);
        return new XmlTypeAndValue(
                convertGreece(valuesMap.get(KEY.PAYER_MS_KEY)),
                valuesMap.get(KEY.PAYER_MS_SOURCE_KEY)
        );
    }

    private void parsePspRole(final XmlReportedTransaction xmlReportedTransaction) throws XMLStreamException {
        final Optional<Map<String, String>> valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.PSP_ROLE_QNAME);
        if (valuesMap.isPresent()) {
            final String pspRoleType = valuesMap.get().get(KEY.PSP_ROLE_TYPE_KEY);
            final String pspRoleOther = valuesMap.get().get(KEY.PSP_ROLE_OTHER_KEY);
            xmlReportedTransaction.setPspRoleType(pspRoleType);
            if (isNotBlank(pspRoleOther)) {
                final int index = this.otherPspRoles.indexOf(pspRoleOther);
                if (index == -1) {
                    this.otherPspRoles.add(pspRoleOther);
                    xmlReportedTransaction.setPspRoleOtherId(this.otherPspRoles.size());
                } else {
                    xmlReportedTransaction.setPspRoleOtherId(index + 1);
                }
            }
        }
    }
}
